% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/quantreg-rq.R
\name{glance.rq}
\alias{glance.rq}
\title{Glance at a(n) rq object}
\usage{
\method{glance}{rq}(x, ...)
}
\arguments{
\item{x}{An \code{rq} object returned from \code{\link[quantreg:rq]{quantreg::rq()}}.}

\item{...}{Additional arguments. Not used. Needed to match generic
signature only. \strong{Cautionary note:} Misspelled arguments will be
absorbed in \code{...}, where they will be ignored. If the misspelled
argument has a default value, the default value will be used.
For example, if you pass \code{conf.lvel = 0.9}, all computation will
proceed using \code{conf.level = 0.95}. Two exceptions here are:
\itemize{
\item \code{tidy()} methods will warn when supplied an \code{exponentiate} argument if
it will be ignored.
\item \code{augment()} methods will warn when supplied a \code{newdata} argument if it
will be ignored.
}}
}
\description{
Glance accepts a model object and returns a \code{\link[tibble:tibble]{tibble::tibble()}}
with exactly one row of model summaries. The summaries are typically
goodness of fit measures, p-values for hypothesis tests on residuals,
or model convergence information.

Glance never returns information from the original call to the modeling
function. This includes the name of the modeling function or any
arguments passed to the modeling function.

Glance does not calculate summary measures. Rather, it farms out these
computations to appropriate methods and gathers the results together.
Sometimes a goodness of fit measure will be undefined. In these cases
the measure will be reported as \code{NA}.

Glance returns the same number of columns regardless of whether the
model matrix is rank-deficient or not. If so, entries in columns
that no longer have a well-defined value are filled in with an \code{NA}
of the appropriate type.
}
\details{
Only models with a single \code{tau} value may be passed.
For multiple values, please use a \code{\link[purrr:map]{purrr::map()}} workflow instead, e.g.

\if{html}{\out{<div class="sourceCode">}}\preformatted{taus |>
  map(function(tau_val) rq(y ~ x, tau = tau_val)) |>
  map_dfr(glance)
}\if{html}{\out{</div>}}
}
\examples{
\dontshow{if (rlang::is_installed("quantreg")) withAutoprint(\{ # examplesIf}

# load modeling library and data
library(quantreg)

data(stackloss)

# median (l1) regression fit for the stackloss data.
mod1 <- rq(stack.loss ~ stack.x, .5)

# weighted sample median
mod2 <- rq(rnorm(50) ~ 1, weights = runif(50))

# summarize model fit with tidiers
tidy(mod1)
glance(mod1)
augment(mod1)

tidy(mod2)
glance(mod2)
augment(mod2)

# varying tau to generate an rqs object
mod3 <- rq(stack.loss ~ stack.x, tau = c(.25, .5))

tidy(mod3)
augment(mod3)

# glance cannot handle rqs objects like `mod3`--use a purrr
# `map`-based workflow instead
\dontshow{\}) # examplesIf}
}
\seealso{
\code{\link[=glance]{glance()}}, \code{\link[quantreg:rq]{quantreg::rq()}}

Other quantreg tidiers: 
\code{\link{augment.nlrq}()},
\code{\link{augment.rq}()},
\code{\link{augment.rqs}()},
\code{\link{glance.nlrq}()},
\code{\link{tidy.nlrq}()},
\code{\link{tidy.rq}()},
\code{\link{tidy.rqs}()}
}
\concept{quantreg tidiers}
\value{
A \code{\link[tibble:tibble]{tibble::tibble()}} with exactly one row and columns:
  \item{AIC}{Akaike's Information Criterion for the model.}
  \item{BIC}{Bayesian Information Criterion for the model.}
  \item{df.residual}{Residual degrees of freedom.}
  \item{logLik}{The log-likelihood of the model. [stats::logLik()] may be a useful reference.}
  \item{tau}{Quantile.}

}
