/*
 *
 *  Copyright (C) 2022-2024, OFFIS e.V.
 *  All rights reserved.  See COPYRIGHT file for details.
 *
 *  This software and supporting documentation were developed by
 *
 *    OFFIS e.V.
 *    R&D Division Health
 *    Escherweg 2
 *    D-26121 Oldenburg, Germany
 *
 *
 *  Module:  oficonv
 *
 *  Author:  Marco Eichelberg
 *
 *  Purpose: ISO 8859 related tables for oficonv unit tests
 *
 */

#ifndef TABLES_ISO8859_H
#define TABLES_ISO8859_H

// printable ASCII characters (ISO 646). These are the same in all ISO 8859 alphabets.
unsigned char char_20_to_7e[] = {
  0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
  0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
  0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
  0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
  0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
  0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e
};

// characters A0-FF. Some ISO 8859 alphabets, in particular 8859-1, use of all these.
unsigned char char_a0_to_ff[] = {
  0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
  0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
  0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
  0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
  0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
  0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff
};

// ISO 8859-3 has a few gaps in the A0-FF range
unsigned char iso8859_3_char_a0_to_ff[] = {
  0xa0, 0xa1, 0xa2, 0xa3, 0xa4,       0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad,       0xaf,
  0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd,       0xbf,
  0xc0, 0xc1, 0xc2,       0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
        0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
  0xe0, 0xe1, 0xe2,       0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
        0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff
};

// ISO 8859-6 has a few gaps in the A0-FF range
unsigned char iso8859_6_char_a0_to_ff[] = {
  0xa0,                   0xa4,                                           0xac, 0xad,
                                                                    0xbb,                   0xbf,
        0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
  0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda,
  0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
  0xf0, 0xf1, 0xf2
};

// ISO 8859-7 has a few gaps in the A0-FF range
unsigned char iso8859_7_char_a0_to_ff[] = {
  0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad,       0xaf,
  0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
  0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
  0xd0, 0xd1,       0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
  0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
  0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe
};

// ISO 8859-8 has a few gaps in the A0-FF range
unsigned char iso8859_8_char_a0_to_ff[] = {
  0xa0,       0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
  0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe,
                                                                                            0xdf,
  0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
  0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa,             0xfd, 0xfe
};

// ISO 8859-11 has a few gaps in the A0-FF range
unsigned char iso8859_11_char_a0_to_ff[] = {
  0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
  0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
  0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
  0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda,                         0xdf,
  0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
  0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb
};

// All well-defined ISO 8859-1 characters in the range A0-FF converted to UTF-8
unsigned char iso8859_1_to_utf8_a0_to_ff[] = {
  0xc2, 0xa0, 0xc2, 0xa1, 0xc2, 0xa2, 0xc2, 0xa3, 0xc2,
  0xa4, 0xc2, 0xa5, 0xc2, 0xa6, 0xc2, 0xa7, 0xc2, 0xa8, 0xc2, 0xa9, 0xc2,
  0xaa, 0xc2, 0xab, 0xc2, 0xac, 0xc2, 0xad, 0xc2, 0xae, 0xc2, 0xaf, 0xc2,
  0xb0, 0xc2, 0xb1, 0xc2, 0xb2, 0xc2, 0xb3, 0xc2, 0xb4, 0xc2, 0xb5, 0xc2,
  0xb6, 0xc2, 0xb7, 0xc2, 0xb8, 0xc2, 0xb9, 0xc2, 0xba, 0xc2, 0xbb, 0xc2,
  0xbc, 0xc2, 0xbd, 0xc2, 0xbe, 0xc2, 0xbf, 0xc3, 0x80, 0xc3, 0x81, 0xc3,
  0x82, 0xc3, 0x83, 0xc3, 0x84, 0xc3, 0x85, 0xc3, 0x86, 0xc3, 0x87, 0xc3,
  0x88, 0xc3, 0x89, 0xc3, 0x8a, 0xc3, 0x8b, 0xc3, 0x8c, 0xc3, 0x8d, 0xc3,
  0x8e, 0xc3, 0x8f, 0xc3, 0x90, 0xc3, 0x91, 0xc3, 0x92, 0xc3, 0x93, 0xc3,
  0x94, 0xc3, 0x95, 0xc3, 0x96, 0xc3, 0x97, 0xc3, 0x98, 0xc3, 0x99, 0xc3,
  0x9a, 0xc3, 0x9b, 0xc3, 0x9c, 0xc3, 0x9d, 0xc3, 0x9e, 0xc3, 0x9f, 0xc3,
  0xa0, 0xc3, 0xa1, 0xc3, 0xa2, 0xc3, 0xa3, 0xc3, 0xa4, 0xc3, 0xa5, 0xc3,
  0xa6, 0xc3, 0xa7, 0xc3, 0xa8, 0xc3, 0xa9, 0xc3, 0xaa, 0xc3, 0xab, 0xc3,
  0xac, 0xc3, 0xad, 0xc3, 0xae, 0xc3, 0xaf, 0xc3, 0xb0, 0xc3, 0xb1, 0xc3,
  0xb2, 0xc3, 0xb3, 0xc3, 0xb4, 0xc3, 0xb5, 0xc3, 0xb6, 0xc3, 0xb7, 0xc3,
  0xb8, 0xc3, 0xb9, 0xc3, 0xba, 0xc3, 0xbb, 0xc3, 0xbc, 0xc3, 0xbd, 0xc3,
  0xbe, 0xc3, 0xbf
};

// All well-defined ISO 8859-2 characters in the range A0-FF converted to UTF-8
unsigned char iso8859_2_to_utf8_a0_to_ff[] = {
  0xc2, 0xa0, 0xc4, 0x84, 0xcb, 0x98, 0xc5, 0x81, 0xc2, 0xa4, 0xc4, 0xbd,
  0xc5, 0x9a, 0xc2, 0xa7, 0xc2, 0xa8, 0xc5, 0xa0, 0xc5, 0x9e, 0xc5, 0xa4,
  0xc5, 0xb9, 0xc2, 0xad, 0xc5, 0xbd, 0xc5, 0xbb, 0xc2, 0xb0, 0xc4, 0x85,
  0xcb, 0x9b, 0xc5, 0x82, 0xc2, 0xb4, 0xc4, 0xbe, 0xc5, 0x9b, 0xcb, 0x87,
  0xc2, 0xb8, 0xc5, 0xa1, 0xc5, 0x9f, 0xc5, 0xa5, 0xc5, 0xba, 0xcb, 0x9d,
  0xc5, 0xbe, 0xc5, 0xbc, 0xc5, 0x94, 0xc3, 0x81, 0xc3, 0x82, 0xc4, 0x82,
  0xc3, 0x84, 0xc4, 0xb9, 0xc4, 0x86, 0xc3, 0x87, 0xc4, 0x8c, 0xc3, 0x89,
  0xc4, 0x98, 0xc3, 0x8b, 0xc4, 0x9a, 0xc3, 0x8d, 0xc3, 0x8e, 0xc4, 0x8e,
  0xc4, 0x90, 0xc5, 0x83, 0xc5, 0x87, 0xc3, 0x93, 0xc3, 0x94, 0xc5, 0x90,
  0xc3, 0x96, 0xc3, 0x97, 0xc5, 0x98, 0xc5, 0xae, 0xc3, 0x9a, 0xc5, 0xb0,
  0xc3, 0x9c, 0xc3, 0x9d, 0xc5, 0xa2, 0xc3, 0x9f, 0xc5, 0x95, 0xc3, 0xa1,
  0xc3, 0xa2, 0xc4, 0x83, 0xc3, 0xa4, 0xc4, 0xba, 0xc4, 0x87, 0xc3, 0xa7,
  0xc4, 0x8d, 0xc3, 0xa9, 0xc4, 0x99, 0xc3, 0xab, 0xc4, 0x9b, 0xc3, 0xad,
  0xc3, 0xae, 0xc4, 0x8f, 0xc4, 0x91, 0xc5, 0x84, 0xc5, 0x88, 0xc3, 0xb3,
  0xc3, 0xb4, 0xc5, 0x91, 0xc3, 0xb6, 0xc3, 0xb7, 0xc5, 0x99, 0xc5, 0xaf,
  0xc3, 0xba, 0xc5, 0xb1, 0xc3, 0xbc, 0xc3, 0xbd, 0xc5, 0xa3, 0xcb, 0x99
};

// All well-defined ISO 8859-3 characters in the range A0-FF converted to UTF-8
unsigned char iso8859_3_to_utf8_a0_to_ff[] = {
  0xc2, 0xa0, 0xc4, 0xa6, 0xcb, 0x98, 0xc2, 0xa3, 0xc2, 0xa4, 0xc4, 0xa4,
  0xc2, 0xa7, 0xc2, 0xa8, 0xc4, 0xb0, 0xc5, 0x9e, 0xc4, 0x9e, 0xc4, 0xb4,
  0xc2, 0xad, 0xc5, 0xbb, 0xc2, 0xb0, 0xc4, 0xa7, 0xc2, 0xb2, 0xc2, 0xb3,
  0xc2, 0xb4, 0xc2, 0xb5, 0xc4, 0xa5, 0xc2, 0xb7, 0xc2, 0xb8, 0xc4, 0xb1,
  0xc5, 0x9f, 0xc4, 0x9f, 0xc4, 0xb5, 0xc2, 0xbd, 0xc5, 0xbc, 0xc3, 0x80,
  0xc3, 0x81, 0xc3, 0x82, 0xc3, 0x84, 0xc4, 0x8a, 0xc4, 0x88, 0xc3, 0x87,
  0xc3, 0x88, 0xc3, 0x89, 0xc3, 0x8a, 0xc3, 0x8b, 0xc3, 0x8c, 0xc3, 0x8d,
  0xc3, 0x8e, 0xc3, 0x8f, 0xc3, 0x91, 0xc3, 0x92, 0xc3, 0x93, 0xc3, 0x94,
  0xc4, 0xa0, 0xc3, 0x96, 0xc3, 0x97, 0xc4, 0x9c, 0xc3, 0x99, 0xc3, 0x9a,
  0xc3, 0x9b, 0xc3, 0x9c, 0xc5, 0xac, 0xc5, 0x9c, 0xc3, 0x9f, 0xc3, 0xa0,
  0xc3, 0xa1, 0xc3, 0xa2, 0xc3, 0xa4, 0xc4, 0x8b, 0xc4, 0x89, 0xc3, 0xa7,
  0xc3, 0xa8, 0xc3, 0xa9, 0xc3, 0xaa, 0xc3, 0xab, 0xc3, 0xac, 0xc3, 0xad,
  0xc3, 0xae, 0xc3, 0xaf, 0xc3, 0xb1, 0xc3, 0xb2, 0xc3, 0xb3, 0xc3, 0xb4,
  0xc4, 0xa1, 0xc3, 0xb6, 0xc3, 0xb7, 0xc4, 0x9d, 0xc3, 0xb9, 0xc3, 0xba,
  0xc3, 0xbb, 0xc3, 0xbc, 0xc5, 0xad, 0xc5, 0x9d, 0xcb, 0x99
};

// All well-defined ISO 8859-4 characters in the range A0-FF converted to UTF-8
unsigned char iso8859_4_to_utf8_a0_to_ff[] = {
  0xc2, 0xa0, 0xc4, 0x84, 0xc4, 0xb8, 0xc5, 0x96, 0xc2, 0xa4, 0xc4, 0xa8,
  0xc4, 0xbb, 0xc2, 0xa7, 0xc2, 0xa8, 0xc5, 0xa0, 0xc4, 0x92, 0xc4, 0xa2,
  0xc5, 0xa6, 0xc2, 0xad, 0xc5, 0xbd, 0xc2, 0xaf, 0xc2, 0xb0, 0xc4, 0x85,
  0xcb, 0x9b, 0xc5, 0x97, 0xc2, 0xb4, 0xc4, 0xa9, 0xc4, 0xbc, 0xcb, 0x87,
  0xc2, 0xb8, 0xc5, 0xa1, 0xc4, 0x93, 0xc4, 0xa3, 0xc5, 0xa7, 0xc5, 0x8a,
  0xc5, 0xbe, 0xc5, 0x8b, 0xc4, 0x80, 0xc3, 0x81, 0xc3, 0x82, 0xc3, 0x83,
  0xc3, 0x84, 0xc3, 0x85, 0xc3, 0x86, 0xc4, 0xae, 0xc4, 0x8c, 0xc3, 0x89,
  0xc4, 0x98, 0xc3, 0x8b, 0xc4, 0x96, 0xc3, 0x8d, 0xc3, 0x8e, 0xc4, 0xaa,
  0xc4, 0x90, 0xc5, 0x85, 0xc5, 0x8c, 0xc4, 0xb6, 0xc3, 0x94, 0xc3, 0x95,
  0xc3, 0x96, 0xc3, 0x97, 0xc3, 0x98, 0xc5, 0xb2, 0xc3, 0x9a, 0xc3, 0x9b,
  0xc3, 0x9c, 0xc5, 0xa8, 0xc5, 0xaa, 0xc3, 0x9f, 0xc4, 0x81, 0xc3, 0xa1,
  0xc3, 0xa2, 0xc3, 0xa3, 0xc3, 0xa4, 0xc3, 0xa5, 0xc3, 0xa6, 0xc4, 0xaf,
  0xc4, 0x8d, 0xc3, 0xa9, 0xc4, 0x99, 0xc3, 0xab, 0xc4, 0x97, 0xc3, 0xad,
  0xc3, 0xae, 0xc4, 0xab, 0xc4, 0x91, 0xc5, 0x86, 0xc5, 0x8d, 0xc4, 0xb7,
  0xc3, 0xb4, 0xc3, 0xb5, 0xc3, 0xb6, 0xc3, 0xb7, 0xc3, 0xb8, 0xc5, 0xb3,
  0xc3, 0xba, 0xc3, 0xbb, 0xc3, 0xbc, 0xc5, 0xa9, 0xc5, 0xab, 0xcb, 0x99
};

// All well-defined ISO 8859-5 characters in the range A0-FF converted to UTF-8
unsigned char iso8859_5_to_utf8_a0_to_ff[] = {
  0xc2, 0xa0, 0xd0, 0x81, 0xd0, 0x82, 0xd0, 0x83, 0xd0,
  0x84, 0xd0, 0x85, 0xd0, 0x86, 0xd0, 0x87, 0xd0, 0x88, 0xd0, 0x89, 0xd0,
  0x8a, 0xd0, 0x8b, 0xd0, 0x8c, 0xc2, 0xad, 0xd0, 0x8e, 0xd0, 0x8f, 0xd0,
  0x90, 0xd0, 0x91, 0xd0, 0x92, 0xd0, 0x93, 0xd0, 0x94, 0xd0, 0x95, 0xd0,
  0x96, 0xd0, 0x97, 0xd0, 0x98, 0xd0, 0x99, 0xd0, 0x9a, 0xd0, 0x9b, 0xd0,
  0x9c, 0xd0, 0x9d, 0xd0, 0x9e, 0xd0, 0x9f, 0xd0, 0xa0, 0xd0, 0xa1, 0xd0,
  0xa2, 0xd0, 0xa3, 0xd0, 0xa4, 0xd0, 0xa5, 0xd0, 0xa6, 0xd0, 0xa7, 0xd0,
  0xa8, 0xd0, 0xa9, 0xd0, 0xaa, 0xd0, 0xab, 0xd0, 0xac, 0xd0, 0xad, 0xd0,
  0xae, 0xd0, 0xaf, 0xd0, 0xb0, 0xd0, 0xb1, 0xd0, 0xb2, 0xd0, 0xb3, 0xd0,
  0xb4, 0xd0, 0xb5, 0xd0, 0xb6, 0xd0, 0xb7, 0xd0, 0xb8, 0xd0, 0xb9, 0xd0,
  0xba, 0xd0, 0xbb, 0xd0, 0xbc, 0xd0, 0xbd, 0xd0, 0xbe, 0xd0, 0xbf, 0xd1,
  0x80, 0xd1, 0x81, 0xd1, 0x82, 0xd1, 0x83, 0xd1, 0x84, 0xd1, 0x85, 0xd1,
  0x86, 0xd1, 0x87, 0xd1, 0x88, 0xd1, 0x89, 0xd1, 0x8a, 0xd1, 0x8b, 0xd1,
  0x8c, 0xd1, 0x8d, 0xd1, 0x8e, 0xd1, 0x8f, 0xe2, 0x84, 0x96, 0xd1, 0x91,
  0xd1, 0x92, 0xd1, 0x93, 0xd1, 0x94, 0xd1, 0x95, 0xd1, 0x96, 0xd1, 0x97,
  0xd1, 0x98, 0xd1, 0x99, 0xd1, 0x9a, 0xd1, 0x9b, 0xd1, 0x9c, 0xc2, 0xa7,
  0xd1, 0x9e, 0xd1, 0x9f
};

// All well-defined ISO 8859-6 characters in the range A0-FF converted to UTF-8
unsigned char iso8859_6_to_utf8_a0_to_ff[] = {
  0xc2, 0xa0, 0xc2, 0xa4, 0xd8, 0x8c, 0xc2, 0xad, 0xd8, 0x9b, 0xd8,
  0x9f, 0xd8, 0xa1, 0xd8, 0xa2, 0xd8, 0xa3, 0xd8, 0xa4, 0xd8, 0xa5, 0xd8,
  0xa6, 0xd8, 0xa7, 0xd8, 0xa8, 0xd8, 0xa9, 0xd8, 0xaa, 0xd8, 0xab, 0xd8,
  0xac, 0xd8, 0xad, 0xd8, 0xae, 0xd8, 0xaf, 0xd8, 0xb0, 0xd8, 0xb1, 0xd8,
  0xb2, 0xd8, 0xb3, 0xd8, 0xb4, 0xd8, 0xb5, 0xd8, 0xb6, 0xd8, 0xb7, 0xd8,
  0xb8, 0xd8, 0xb9, 0xd8, 0xba, 0xd9, 0x80, 0xd9, 0x81, 0xd9, 0x82, 0xd9,
  0x83, 0xd9, 0x84, 0xd9, 0x85, 0xd9, 0x86, 0xd9, 0x87, 0xd9, 0x88, 0xd9,
  0x89, 0xd9, 0x8a, 0xd9, 0x8b, 0xd9, 0x8c, 0xd9, 0x8d, 0xd9, 0x8e, 0xd9,
  0x8f, 0xd9, 0x90, 0xd9, 0x91, 0xd9, 0x92
};

// All well-defined ISO 8859-7 characters in the range A0-FF converted to UTF-8
unsigned char iso8859_7_to_utf8_a0_to_ff[] = {
  0xc2, 0xa0, 0xe2, 0x80, 0x98, 0xe2, 0x80, 0x99, 0xc2, 0xa3, 0xe2, 0x82,
  0xac, 0xe2, 0x82, 0xaf, 0xc2, 0xa6, 0xc2, 0xa7, 0xc2, 0xa8, 0xc2, 0xa9,
  0xcd, 0xba, 0xc2, 0xab, 0xc2, 0xac, 0xc2, 0xad, 0xe2, 0x80, 0x95, 0xc2,
  0xb0, 0xc2, 0xb1, 0xc2, 0xb2, 0xc2, 0xb3, 0xce, 0x84, 0xce, 0x85, 0xce,
  0x86, 0xc2, 0xb7, 0xce, 0x88, 0xce, 0x89, 0xce, 0x8a, 0xc2, 0xbb, 0xce,
  0x8c, 0xc2, 0xbd, 0xce, 0x8e, 0xce, 0x8f, 0xce, 0x90, 0xce, 0x91, 0xce,
  0x92, 0xce, 0x93, 0xce, 0x94, 0xce, 0x95, 0xce, 0x96, 0xce, 0x97, 0xce,
  0x98, 0xce, 0x99, 0xce, 0x9a, 0xce, 0x9b, 0xce, 0x9c, 0xce, 0x9d, 0xce,
  0x9e, 0xce, 0x9f, 0xce, 0xa0, 0xce, 0xa1, 0xce, 0xa3, 0xce, 0xa4, 0xce,
  0xa5, 0xce, 0xa6, 0xce, 0xa7, 0xce, 0xa8, 0xce, 0xa9, 0xce, 0xaa, 0xce,
  0xab, 0xce, 0xac, 0xce, 0xad, 0xce, 0xae, 0xce, 0xaf, 0xce, 0xb0, 0xce,
  0xb1, 0xce, 0xb2, 0xce, 0xb3, 0xce, 0xb4, 0xce, 0xb5, 0xce, 0xb6, 0xce,
  0xb7, 0xce, 0xb8, 0xce, 0xb9, 0xce, 0xba, 0xce, 0xbb, 0xce, 0xbc, 0xce,
  0xbd, 0xce, 0xbe, 0xce, 0xbf, 0xcf, 0x80, 0xcf, 0x81, 0xcf, 0x82, 0xcf,
  0x83, 0xcf, 0x84, 0xcf, 0x85, 0xcf, 0x86, 0xcf, 0x87, 0xcf, 0x88, 0xcf,
  0x89, 0xcf, 0x8a, 0xcf, 0x8b, 0xcf, 0x8c, 0xcf, 0x8d, 0xcf, 0x8e
};

// All well-defined ISO 8859-8 characters in the range A0-FF converted to UTF-8
unsigned char iso8859_8_to_utf8_a0_to_ff[] = {
  0xc2, 0xa0, 0xc2, 0xa2, 0xc2, 0xa3, 0xc2, 0xa4, 0xc2, 0xa5, 0xc2, 0xa6,
  0xc2, 0xa7, 0xc2, 0xa8, 0xc2, 0xa9, 0xc3, 0x97, 0xc2, 0xab, 0xc2, 0xac,
  0xc2, 0xad, 0xc2, 0xae, 0xc2, 0xaf, 0xc2, 0xb0, 0xc2, 0xb1, 0xc2, 0xb2,
  0xc2, 0xb3, 0xc2, 0xb4, 0xc2, 0xb5, 0xc2, 0xb6, 0xc2, 0xb7, 0xc2, 0xb8,
  0xc2, 0xb9, 0xc3, 0xb7, 0xc2, 0xbb, 0xc2, 0xbc, 0xc2, 0xbd, 0xc2, 0xbe,
  0xe2, 0x80, 0x97, 0xd7, 0x90, 0xd7, 0x91, 0xd7, 0x92, 0xd7, 0x93, 0xd7,
  0x94, 0xd7, 0x95, 0xd7, 0x96, 0xd7, 0x97, 0xd7, 0x98, 0xd7, 0x99, 0xd7,
  0x9a, 0xd7, 0x9b, 0xd7, 0x9c, 0xd7, 0x9d, 0xd7, 0x9e, 0xd7, 0x9f, 0xd7,
  0xa0, 0xd7, 0xa1, 0xd7, 0xa2, 0xd7, 0xa3, 0xd7, 0xa4, 0xd7, 0xa5, 0xd7,
  0xa6, 0xd7, 0xa7, 0xd7, 0xa8, 0xd7, 0xa9, 0xd7, 0xaa, 0xe2, 0x80, 0x8e,
  0xe2, 0x80, 0x8f
};

// All well-defined ISO 8859-9 characters in the range A0-FF converted to UTF-8
unsigned char iso8859_9_to_utf8_a0_to_ff[] = {
  0xc2, 0xa0, 0xc2, 0xa1, 0xc2, 0xa2, 0xc2, 0xa3, 0xc2, 0xa4, 0xc2, 0xa5,
  0xc2, 0xa6, 0xc2, 0xa7, 0xc2, 0xa8, 0xc2, 0xa9, 0xc2, 0xaa, 0xc2, 0xab,
  0xc2, 0xac, 0xc2, 0xad, 0xc2, 0xae, 0xc2, 0xaf, 0xc2, 0xb0, 0xc2, 0xb1,
  0xc2, 0xb2, 0xc2, 0xb3, 0xc2, 0xb4, 0xc2, 0xb5, 0xc2, 0xb6, 0xc2, 0xb7,
  0xc2, 0xb8, 0xc2, 0xb9, 0xc2, 0xba, 0xc2, 0xbb, 0xc2, 0xbc, 0xc2, 0xbd,
  0xc2, 0xbe, 0xc2, 0xbf, 0xc3, 0x80, 0xc3, 0x81, 0xc3, 0x82, 0xc3, 0x83,
  0xc3, 0x84, 0xc3, 0x85, 0xc3, 0x86, 0xc3, 0x87, 0xc3, 0x88, 0xc3, 0x89,
  0xc3, 0x8a, 0xc3, 0x8b, 0xc3, 0x8c, 0xc3, 0x8d, 0xc3, 0x8e, 0xc3, 0x8f,
  0xc4, 0x9e, 0xc3, 0x91, 0xc3, 0x92, 0xc3, 0x93, 0xc3, 0x94, 0xc3, 0x95,
  0xc3, 0x96, 0xc3, 0x97, 0xc3, 0x98, 0xc3, 0x99, 0xc3, 0x9a, 0xc3, 0x9b,
  0xc3, 0x9c, 0xc4, 0xb0, 0xc5, 0x9e, 0xc3, 0x9f, 0xc3, 0xa0, 0xc3, 0xa1,
  0xc3, 0xa2, 0xc3, 0xa3, 0xc3, 0xa4, 0xc3, 0xa5, 0xc3, 0xa6, 0xc3, 0xa7,
  0xc3, 0xa8, 0xc3, 0xa9, 0xc3, 0xaa, 0xc3, 0xab, 0xc3, 0xac, 0xc3, 0xad,
  0xc3, 0xae, 0xc3, 0xaf, 0xc4, 0x9f, 0xc3, 0xb1, 0xc3, 0xb2, 0xc3, 0xb3,
  0xc3, 0xb4, 0xc3, 0xb5, 0xc3, 0xb6, 0xc3, 0xb7, 0xc3, 0xb8, 0xc3, 0xb9,
  0xc3, 0xba, 0xc3, 0xbb, 0xc3, 0xbc, 0xc4, 0xb1, 0xc5, 0x9f, 0xc3, 0xbf
};

// All well-defined ISO 8859-11 characters in the range A0-FF converted to UTF-8
unsigned char iso8859_11_to_utf8_a0_to_ff[] = {
  0xc2, 0xa0, 0xe0, 0xb8, 0x81, 0xe0, 0xb8, 0x82, 0xe0, 0xb8, 0x83, 0xe0,
  0xb8, 0x84, 0xe0, 0xb8, 0x85, 0xe0, 0xb8, 0x86, 0xe0, 0xb8, 0x87, 0xe0,
  0xb8, 0x88, 0xe0, 0xb8, 0x89, 0xe0, 0xb8, 0x8a, 0xe0, 0xb8, 0x8b, 0xe0,
  0xb8, 0x8c, 0xe0, 0xb8, 0x8d, 0xe0, 0xb8, 0x8e, 0xe0, 0xb8, 0x8f, 0xe0,
  0xb8, 0x90, 0xe0, 0xb8, 0x91, 0xe0, 0xb8, 0x92, 0xe0, 0xb8, 0x93, 0xe0,
  0xb8, 0x94, 0xe0, 0xb8, 0x95, 0xe0, 0xb8, 0x96, 0xe0, 0xb8, 0x97, 0xe0,
  0xb8, 0x98, 0xe0, 0xb8, 0x99, 0xe0, 0xb8, 0x9a, 0xe0, 0xb8, 0x9b, 0xe0,
  0xb8, 0x9c, 0xe0, 0xb8, 0x9d, 0xe0, 0xb8, 0x9e, 0xe0, 0xb8, 0x9f, 0xe0,
  0xb8, 0xa0, 0xe0, 0xb8, 0xa1, 0xe0, 0xb8, 0xa2, 0xe0, 0xb8, 0xa3, 0xe0,
  0xb8, 0xa4, 0xe0, 0xb8, 0xa5, 0xe0, 0xb8, 0xa6, 0xe0, 0xb8, 0xa7, 0xe0,
  0xb8, 0xa8, 0xe0, 0xb8, 0xa9, 0xe0, 0xb8, 0xaa, 0xe0, 0xb8, 0xab, 0xe0,
  0xb8, 0xac, 0xe0, 0xb8, 0xad, 0xe0, 0xb8, 0xae, 0xe0, 0xb8, 0xaf, 0xe0,
  0xb8, 0xb0, 0xe0, 0xb8, 0xb1, 0xe0, 0xb8, 0xb2, 0xe0, 0xb8, 0xb3, 0xe0,
  0xb8, 0xb4, 0xe0, 0xb8, 0xb5, 0xe0, 0xb8, 0xb6, 0xe0, 0xb8, 0xb7, 0xe0,
  0xb8, 0xb8, 0xe0, 0xb8, 0xb9, 0xe0, 0xb8, 0xba, 0xe0, 0xb8, 0xbf, 0xe0,
  0xb9, 0x80, 0xe0, 0xb9, 0x81, 0xe0, 0xb9, 0x82, 0xe0, 0xb9, 0x83, 0xe0,
  0xb9, 0x84, 0xe0, 0xb9, 0x85, 0xe0, 0xb9, 0x86, 0xe0, 0xb9, 0x87, 0xe0,
  0xb9, 0x88, 0xe0, 0xb9, 0x89, 0xe0, 0xb9, 0x8a, 0xe0, 0xb9, 0x8b, 0xe0,
  0xb9, 0x8c, 0xe0, 0xb9, 0x8d, 0xe0, 0xb9, 0x8e, 0xe0, 0xb9, 0x8f, 0xe0,
  0xb9, 0x90, 0xe0, 0xb9, 0x91, 0xe0, 0xb9, 0x92, 0xe0, 0xb9, 0x93, 0xe0,
  0xb9, 0x94, 0xe0, 0xb9, 0x95, 0xe0, 0xb9, 0x96, 0xe0, 0xb9, 0x97, 0xe0,
  0xb9, 0x98, 0xe0, 0xb9, 0x99, 0xe0, 0xb9, 0x9a, 0xe0, 0xb9, 0x9b
};

// All well-defined ISO 8859-15 characters in the range A0-FF converted to UTF-8
unsigned char iso8859_15_to_utf8_a0_to_ff[] = {
  0xc2, 0xa0, 0xc2, 0xa1, 0xc2, 0xa2, 0xc2, 0xa3, 0xe2, 0x82, 0xac, 0xc2,
  0xa5, 0xc5, 0xa0, 0xc2, 0xa7, 0xc5, 0xa1, 0xc2, 0xa9, 0xc2, 0xaa, 0xc2,
  0xab, 0xc2, 0xac, 0xc2, 0xad, 0xc2, 0xae, 0xc2, 0xaf, 0xc2, 0xb0, 0xc2,
  0xb1, 0xc2, 0xb2, 0xc2, 0xb3, 0xc5, 0xbd, 0xc2, 0xb5, 0xc2, 0xb6, 0xc2,
  0xb7, 0xc5, 0xbe, 0xc2, 0xb9, 0xc2, 0xba, 0xc2, 0xbb, 0xc5, 0x92, 0xc5,
  0x93, 0xc5, 0xb8, 0xc2, 0xbf, 0xc3, 0x80, 0xc3, 0x81, 0xc3, 0x82, 0xc3,
  0x83, 0xc3, 0x84, 0xc3, 0x85, 0xc3, 0x86, 0xc3, 0x87, 0xc3, 0x88, 0xc3,
  0x89, 0xc3, 0x8a, 0xc3, 0x8b, 0xc3, 0x8c, 0xc3, 0x8d, 0xc3, 0x8e, 0xc3,
  0x8f, 0xc3, 0x90, 0xc3, 0x91, 0xc3, 0x92, 0xc3, 0x93, 0xc3, 0x94, 0xc3,
  0x95, 0xc3, 0x96, 0xc3, 0x97, 0xc3, 0x98, 0xc3, 0x99, 0xc3, 0x9a, 0xc3,
  0x9b, 0xc3, 0x9c, 0xc3, 0x9d, 0xc3, 0x9e, 0xc3, 0x9f, 0xc3, 0xa0, 0xc3,
  0xa1, 0xc3, 0xa2, 0xc3, 0xa3, 0xc3, 0xa4, 0xc3, 0xa5, 0xc3, 0xa6, 0xc3,
  0xa7, 0xc3, 0xa8, 0xc3, 0xa9, 0xc3, 0xaa, 0xc3, 0xab, 0xc3, 0xac, 0xc3,
  0xad, 0xc3, 0xae, 0xc3, 0xaf, 0xc3, 0xb0, 0xc3, 0xb1, 0xc3, 0xb2, 0xc3,
  0xb3, 0xc3, 0xb4, 0xc3, 0xb5, 0xc3, 0xb6, 0xc3, 0xb7, 0xc3, 0xb8, 0xc3,
  0xb9, 0xc3, 0xba, 0xc3, 0xbb, 0xc3, 0xbc, 0xc3, 0xbd, 0xc3, 0xbe, 0xc3,
  0xbf
};

// All well-defined ISO 8859-1 characters in the range A0-FF converted to 8859-2
unsigned char iso8859_1_to_iso8859_2[] = {
  0xa0, 0x21, 0x63, 0x6c, 0x62, 0xa4, 0x79, 0x65, 0x6e, 0x7c, 0xa7, 0xa8,
  0x28, 0x63, 0x29, 0x61, 0x3c, 0x3c, 0x6e, 0x6f, 0x74, 0xad, 0x28, 0x52,
  0x29, 0x3f, 0xb0, 0x2b, 0x2f, 0x2d, 0x5e, 0x32, 0x5e, 0x33, 0xb4, 0x75,
  0x50, 0x2e, 0xb8, 0x5e, 0x31, 0x6f, 0x3e, 0x3e, 0x3f, 0x3f, 0x3f, 0x3f,
  0x60, 0x41, 0xc1, 0xc2, 0x7e, 0x41, 0xc4, 0x41, 0x41, 0x45, 0xc7, 0x60,
  0x45, 0xc9, 0x5e, 0x45, 0xcb, 0x60, 0x49, 0xcd, 0xce, 0x22, 0x49, 0x44,
  0x7e, 0x4e, 0x60, 0x4f, 0xd3, 0xd4, 0x7e, 0x4f, 0xd6, 0xd7, 0x4f, 0x60,
  0x55, 0xda, 0x5e, 0x55, 0xdc, 0xdd, 0x54, 0x68, 0xdf, 0x60, 0x61, 0xe1,
  0xe2, 0x7e, 0x61, 0xe4, 0x61, 0x61, 0x65, 0xe7, 0x60, 0x65, 0xe9, 0x5e,
  0x65, 0xeb, 0x60, 0x69, 0xed, 0xee, 0x22, 0x69, 0x64, 0x7e, 0x6e, 0x60,
  0x6f, 0xf3, 0xf4, 0x7e, 0x6f, 0xf6, 0xf7, 0x6f, 0x60, 0x75, 0xfa, 0x5e,
  0x75, 0xfc, 0xfd, 0x74, 0x68, 0x22, 0x79
};

// All well-defined ISO 8859-1 characters in the range A0-FF converted to 8859-3
unsigned char iso8859_1_to_iso8859_3[] = {
  0xa0, 0x21, 0x63, 0xa3, 0xa4, 0x79, 0x65, 0x6e, 0x7c, 0xa7, 0xa8, 0x28,
  0x63, 0x29, 0x61, 0x3c, 0x3c, 0x6e, 0x6f, 0x74, 0xad, 0x28, 0x52, 0x29,
  0x3f, 0xb0, 0x2b, 0x2f, 0x2d, 0xb2, 0xb3, 0xb4, 0xb5, 0x50, 0xb7, 0xb8,
  0x5e, 0x31, 0x6f, 0x3e, 0x3e, 0x3f, 0xbd, 0x3f, 0x3f, 0xc0, 0xc1, 0xc2,
  0x7e, 0x41, 0xc4, 0x41, 0x41, 0x45, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc,
  0xcd, 0xce, 0xcf, 0x44, 0xd1, 0xd2, 0xd3, 0xd4, 0x7e, 0x4f, 0xd6, 0xd7,
  0x4f, 0xd9, 0xda, 0xdb, 0xdc, 0xb4, 0x59, 0x54, 0x68, 0xdf, 0xe0, 0xe1,
  0xe2, 0x7e, 0x61, 0xe4, 0x61, 0x61, 0x65, 0xe7, 0xe8, 0xe9, 0xea, 0xeb,
  0xec, 0xed, 0xee, 0xef, 0x64, 0xf1, 0xf2, 0xf3, 0xf4, 0x7e, 0x6f, 0xf6,
  0xf7, 0x6f, 0xf9, 0xfa, 0xfb, 0xfc, 0xb4, 0x79, 0x74, 0x68, 0x22, 0x79
};

// All well-defined ISO 8859-1 characters in the range A0-FF converted to 8859-4
unsigned char iso8859_1_to_iso8859_4[] = {
  0xa0, 0x21, 0x63, 0x6c, 0x62, 0xa4, 0x79, 0x65, 0x6e, 0x7c, 0xa7, 0xa8,
  0x28, 0x63, 0x29, 0x61, 0x3c, 0x3c, 0x6e, 0x6f, 0x74, 0xad, 0x28, 0x52,
  0x29, 0xaf, 0xb0, 0x2b, 0x2f, 0x2d, 0x5e, 0x32, 0x5e, 0x33, 0xb4, 0x75,
  0x50, 0x2e, 0xb8, 0x5e, 0x31, 0x6f, 0x3e, 0x3e, 0x3f, 0x3f, 0x3f, 0x3f,
  0x60, 0x41, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0x43, 0x60, 0x45, 0xc9,
  0x5e, 0x45, 0xcb, 0x60, 0x49, 0xcd, 0xce, 0x22, 0x49, 0x44, 0x7e, 0x4e,
  0x60, 0x4f, 0xb4, 0x4f, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0x60, 0x55, 0xda,
  0xdb, 0xdc, 0xb4, 0x59, 0x54, 0x68, 0xdf, 0x60, 0x61, 0xe1, 0xe2, 0xe3,
  0xe4, 0xe5, 0xe6, 0x63, 0x60, 0x65, 0xe9, 0x5e, 0x65, 0xeb, 0x60, 0x69,
  0xed, 0xee, 0x22, 0x69, 0x64, 0x7e, 0x6e, 0x60, 0x6f, 0xb4, 0x6f, 0xf4,
  0xf5, 0xf6, 0xf7, 0xf8, 0x60, 0x75, 0xfa, 0xfb, 0xfc, 0xb4, 0x79, 0x74,
  0x68, 0x22, 0x79
};

// All well-defined ISO 8859-1 characters in the range A0-FF converted to 8859-5
unsigned char iso8859_1_to_iso8859_5[] = {
  0xa0, 0x21, 0x63, 0x6c, 0x62, 0x3f, 0x79, 0x65, 0x6e, 0x7c, 0xfd, 0x22,
  0x28, 0x63, 0x29, 0x61, 0x3c, 0x3c, 0x6e, 0x6f, 0x74, 0xad, 0x28, 0x52,
  0x29, 0x3f, 0x5e, 0x30, 0x2b, 0x2f, 0x2d, 0x5e, 0x32, 0x5e, 0x33, 0x27,
  0x75, 0x50, 0x2e, 0x2c, 0x5e, 0x31, 0x6f, 0x3e, 0x3e, 0x3f, 0x3f, 0x3f,
  0x3f, 0x60, 0x41, 0x27, 0x41, 0x5e, 0x41, 0x7e, 0x41, 0x22, 0x41, 0x41,
  0x41, 0x45, 0x43, 0x60, 0x45, 0x27, 0x45, 0x5e, 0x45, 0x22, 0x45, 0x60,
  0x49, 0x27, 0x49, 0x5e, 0x49, 0x22, 0x49, 0x44, 0x7e, 0x4e, 0x60, 0x4f,
  0x27, 0x4f, 0x5e, 0x4f, 0x7e, 0x4f, 0x22, 0x4f, 0x78, 0x4f, 0x60, 0x55,
  0x27, 0x55, 0x5e, 0x55, 0x22, 0x55, 0x27, 0x59, 0x54, 0x68, 0x73, 0x73,
  0x60, 0x61, 0x27, 0x61, 0x5e, 0x61, 0x7e, 0x61, 0x22, 0x61, 0x61, 0x61,
  0x65, 0x63, 0x60, 0x65, 0x27, 0x65, 0x5e, 0x65, 0x22, 0x65, 0x60, 0x69,
  0x27, 0x69, 0x5e, 0x69, 0x22, 0x69, 0x64, 0x7e, 0x6e, 0x60, 0x6f, 0x27,
  0x6f, 0x5e, 0x6f, 0x7e, 0x6f, 0x22, 0x6f, 0x3a, 0x6f, 0x60, 0x75, 0x27,
  0x75, 0x5e, 0x75, 0x22, 0x75, 0x27, 0x79, 0x74, 0x68, 0x22, 0x79
};

// All well-defined ISO 8859-1 characters in the range A0-FF converted to 8859-9
unsigned char iso8859_1_to_iso8859_9[] = {
  0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab,
  0xac, 0xad, 0xae, 0xaf, 0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
  0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 0xc0, 0xc1, 0xc2, 0xc3,
  0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
  0x44, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb,
  0xdc, 0xb4, 0x59, 0x54, 0x68, 0xdf, 0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5,
  0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 0x64, 0xf1,
  0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xb4,
  0x79, 0x74, 0x68, 0xff
};

#endif
