///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/viewport/Viewport.h>
#include <core/scene/ObjectNode.h>
#include <core/scene/animation/controller/StandardControllers.h>
#include <core/reference/CloneHelper.h>
#include <core/gui/properties/FloatControllerUI.h>
#include <core/gui/properties/VectorControllerUI.h>
#include <core/gui/properties/ColorControllerUI.h>
#include <core/plugins/PluginManager.h>
#include "ColorCodingModifier.h"

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(ColorCodingModifier, AtomsObjectModifierBase)
DEFINE_REFERENCE_FIELD(ColorCodingModifier, FloatController, "StartValue", startValueCtrl)
DEFINE_REFERENCE_FIELD(ColorCodingModifier, FloatController, "EndValue", endValueCtrl)
DEFINE_REFERENCE_FIELD(ColorCodingModifier, ColorCodingGradient, "ColorGradient", _colorGradient)
DEFINE_PROPERTY_FIELD(ColorCodingModifier, "SourceDataChannel", _sourceDataChannel)
DEFINE_PROPERTY_FIELD(ColorCodingModifier, "SourceVectorComponent", _sourceVectorComponent)
SET_PROPERTY_FIELD_LABEL(ColorCodingModifier, startValueCtrl, "Start value")
SET_PROPERTY_FIELD_LABEL(ColorCodingModifier, endValueCtrl, "End value")
SET_PROPERTY_FIELD_LABEL(ColorCodingModifier, _sourceDataChannel, "Data channel")
SET_PROPERTY_FIELD_LABEL(ColorCodingModifier, _sourceVectorComponent, "Channel component")
SET_PROPERTY_FIELD_LABEL(ColorCodingModifier, _colorGradient, "Color gradient")

IMPLEMENT_ABSTRACT_PLUGIN_CLASS(ColorCodingGradient, RefTarget)
IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(ColorCodingHSVGradient, ColorCodingGradient)
IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(ColorCodingGrayscaleGradient, ColorCodingGradient)

/******************************************************************************
* Constructs the modifier object.
******************************************************************************/
ColorCodingModifier::ColorCodingModifier(bool isLoading) : AtomsObjectModifierBase(isLoading), _sourceVectorComponent(0)
{
	INIT_PROPERTY_FIELD(ColorCodingModifier, startValueCtrl);
	INIT_PROPERTY_FIELD(ColorCodingModifier, endValueCtrl);
	INIT_PROPERTY_FIELD(ColorCodingModifier, _colorGradient);
	INIT_PROPERTY_FIELD(ColorCodingModifier, _sourceDataChannel);
	INIT_PROPERTY_FIELD(ColorCodingModifier, _sourceVectorComponent);

	if(!isLoading) {
		_colorGradient = new ColorCodingHSVGradient();
		startValueCtrl = CONTROLLER_MANAGER.createDefaultController<FloatController>();
		endValueCtrl = CONTROLLER_MANAGER.createDefaultController<FloatController>();
	}
}

/******************************************************************************
* Asks the modifier for its validity interval at the given time.
******************************************************************************/
TimeInterval ColorCodingModifier::modifierValidity(TimeTicks time)
{
	TimeInterval interval = TimeForever;
	if(startValueCtrl) startValueCtrl->validityInterval(time, interval);
	if(endValueCtrl) endValueCtrl->validityInterval(time, interval);
	return interval;
}

/******************************************************************************
* This modifies the input object.
******************************************************************************/
EvaluationStatus ColorCodingModifier::modifyAtomsObject(TimeTicks time, TimeInterval& validityInterval)
{
	if(sourceDataChannel().isEmpty())
		throw Exception(tr("Please select a data channel."));

	DataChannel* channel = input()->findDataChannelByName(sourceDataChannel());
	if(!channel)
		throw Exception(tr("The data channel with the name '%1' does not exist in the input object.").arg(sourceDataChannel()));
	if(sourceVectorComponent() >= (int)channel->componentCount())
		throw Exception(tr("The selected vector component is out of range. The data channel '%1' contains only %2 values per atom.").arg(sourceDataChannel()).arg(channel->componentCount()));
	if(!_colorGradient)
		throw Exception(tr("No color gradient has been selected."));

	// Get modifier's parameter values.
	FloatType startValue = 0, endValue = 0;
	if(startValueCtrl) startValueCtrl->getValue(time, startValue, validityInterval);
	if(endValueCtrl) endValueCtrl->getValue(time, endValue, validityInterval);

	// Get the deep copy of the color channel.
	DataChannel* colorChannel = outputStandardChannel(DataChannel::ColorChannel);

	// Iterate over all atoms.
	OVITO_ASSERT(colorChannel->size() == channel->size());
	if(channel->type() == qMetaTypeId<FloatType>()) {
		const FloatType* v = channel->constDataFloat() + sourceVectorComponent();
		Vector3* c = colorChannel->dataVector3();
		for(size_t i = channel->size(); i != 0; i--, ++c, v += channel->componentCount()) {

			// Compute linear interpolation.
			FloatType t;
			if(startValue == endValue) {
				if((*v) == startValue) t = 0.5;
				else if((*v) > startValue) t = 1.0;
				else t = 0.0;
			}
			else t = ((*v) - startValue) / (endValue - startValue);

			// Clamp values.
			if(t < 0) t = 0;
			else if(t > 1) t = 1;

			*c = _colorGradient->valueToColor(t);
		}
	}
	else if(channel->type() == qMetaTypeId<int>()) {
		const int* v = channel->constDataInt() + sourceVectorComponent();
		Vector3* c = colorChannel->dataVector3();
		for(size_t i = channel->size(); i != 0; i--, ++c, v += channel->componentCount()) {

			// Compute linear interpolation.
			FloatType t;
			if(startValue == endValue) {
				if((*v) == startValue) t = 0.5;
				else if((*v) > startValue) t = 1.0;
				else t = 0.0;
			}
			else t = ((*v) - startValue) / (endValue - startValue);

			// Clamp values.
			if(t < 0) t = 0;
			else if(t > 1) t = 1;

			*c = _colorGradient->valueToColor(t);
		}
	}
	else
		throw Exception(tr("The data channel '%1' has an invalid or non-numeric data type.").arg(sourceDataChannel()));

	return EvaluationStatus();
}

/******************************************************************************
* Sets the start and end value to the minimum and maximum value
* in the selected data channel.
******************************************************************************/
bool ColorCodingModifier::adjustRange()
{
	// Determine the minimum and maximum values of the selected data channel.

	// Get the value data channel from the input object.
	PipelineFlowState inputState = getModifierInput();
	AtomsObject::SmartPtr inputObj = dynamic_object_cast<AtomsObject>(inputState.result());
	if(!inputObj) return false;

	DataChannel* channel = inputObj->findDataChannelByName(sourceDataChannel());
	if(!channel) return false;

	if(sourceVectorComponent() >= (int)channel->componentCount()) return false;

	// Iterate over all atoms.
	FloatType maxValue = -FLOATTYPE_MAX;
	FloatType minValue = +FLOATTYPE_MAX;
	if(channel->type() == qMetaTypeId<FloatType>()) {
		const FloatType* v = channel->constDataFloat() + sourceVectorComponent();
		const FloatType* vend = v + (channel->size() * channel->componentCount());
		for(; v != vend; v += channel->componentCount()) {
			if(*v > maxValue) maxValue = *v;
			if(*v < minValue) minValue = *v;
		}
	}
	else if(channel->type() == qMetaTypeId<int>()) {
		const int* v = channel->constDataInt() + sourceVectorComponent();
		const int* vend = v + (channel->size() * channel->componentCount());
		for(; v != vend; v += channel->componentCount()) {
			if(*v > maxValue) maxValue = *v;
			if(*v < minValue) minValue = *v;
		}
	}
	if(minValue == +FLOATTYPE_MAX) return false;

	if(startValueController())
		startValueController()->setCurrentValue(minValue);
	if(endValueController())
		endValueController()->setCurrentValue(maxValue);

	return true;
}


IMPLEMENT_PLUGIN_CLASS(ColorCodingModifierEditor, AtomsObjectModifierEditorBase)

/******************************************************************************
* Sets up the UI widgets of the editor.
******************************************************************************/
void ColorCodingModifierEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
	// Create a rollout.
	QWidget* rollout = createRollout(tr("Color coding"), rolloutParams, "atomviz.modifiers.color_coding");

    // Create the rollout contents.
	QVBoxLayout* layout1 = new QVBoxLayout(rollout);
	layout1->setContentsMargins(4,4,4,4);
#ifndef Q_WS_MAC
	layout1->setSpacing(0);
#endif

	channelList = new QComboBox(rollout);
	layout1->addWidget(new QLabel(tr("Data channel:"), rollout));
	layout1->addWidget(channelList);
	connect(channelList, SIGNAL(activated(int)), this, SLOT(onDataChannelSelected(int)));

	colorGradientList = new QComboBox(rollout);
	layout1->addWidget(new QLabel(tr("Color gradient:"), rollout));
	layout1->addWidget(colorGradientList);
	connect(colorGradientList, SIGNAL(activated(int)), this, SLOT(onColorGradientSelected(int)));
	Q_FOREACH(PluginClassDescriptor* clazz, PLUGIN_MANAGER.listClasses(PLUGINCLASSINFO(ColorCodingGradient))) {
		colorGradientList->addItem(clazz->schematicTitle(), qVariantFromValue((void*)clazz));
	}

	// Update channel list if another modifier has been loaded into the editor.
	connect(this, SIGNAL(contentsReplaced(RefTarget*)), this, SLOT(updateChannelList()));
	// It's the same for the color legend.
	connect(this, SIGNAL(contentsReplaced(RefTarget*)), this, SLOT(updateColorGradient()));

	layout1->addSpacing(10);

	QGridLayout* layout2 = new QGridLayout();
	layout2->setContentsMargins(0,0,0,0);
	layout2->setSpacing(0);
	layout2->setColumnStretch(1, 1);
	layout1->addLayout(layout2);

	// End value parameter.
	FloatControllerUI* endValuePUI = new FloatControllerUI(this, PROPERTY_FIELD_DESCRIPTOR(ColorCodingModifier, endValueCtrl));
	layout2->addWidget(endValuePUI->label(), 0, 0);
	layout2->addLayout(endValuePUI->createFieldLayout(), 0, 1);

	// Insert color legend display.
	colorLegendLabel = new QLabel(rollout);
	colorLegendLabel->setScaledContents(true);
	layout2->addWidget(colorLegendLabel, 1, 1);

	// Start value parameter.
	FloatControllerUI* startValuePUI = new FloatControllerUI(this, PROPERTY_FIELD_DESCRIPTOR(ColorCodingModifier, startValueCtrl));
	layout2->addWidget(startValuePUI->label(), 2, 0);
	layout2->addLayout(startValuePUI->createFieldLayout(), 2, 1);

	layout1->addSpacing(8);
	QPushButton* adjustBtn = new QPushButton(tr("Adjust range"), rollout);
	connect(adjustBtn, SIGNAL(clicked(bool)), this, SLOT(onAdjustRange()));
	layout1->addWidget(adjustBtn);
	layout1->addSpacing(4);
	QPushButton* reverseBtn = new QPushButton(tr("Reverse range"), rollout);
	connect(reverseBtn, SIGNAL(clicked(bool)), this, SLOT(onReverseRange()));
	layout1->addWidget(reverseBtn);

	// Status label.
	layout1->addSpacing(10);
	layout1->addWidget(statusLabel());
}

/******************************************************************************
* Updates the contents of the combo box.
******************************************************************************/
void ColorCodingModifierEditor::updateChannelList()
{
	channelList->clear();

	ColorCodingModifier* mod = static_object_cast<ColorCodingModifier>(editObject());
	if(!mod) return;

	// Obtain the atoms object that is fed into the color coding modifier.
	PipelineFlowState inputState = mod->getModifierInput();
	AtomsObject::SmartPtr inputObj = dynamic_object_cast<AtomsObject>(inputState.result());

	// Populate data channel list from input object.
	int initialIndex = -1;
	if(inputObj) {
		Q_FOREACH(DataChannel* channel, inputObj->dataChannels()) {

			// Channels with a non-numeric data type cannot be used as source for the color coding.
			if(channel->type() != qMetaTypeId<int>() && channel->type() != qMetaTypeId<FloatType>()) continue;

			// Handle scalar and vector channels.
			if(channel->componentNames().empty()) {
				if(mod->sourceDataChannel() == channel->name())
					initialIndex = channelList->count();
				channelList->addItem(channel->name());
				channelList->setItemData(channelList->count()-1, channel->name(), Qt::UserRole+0);
				channelList->setItemData(channelList->count()-1, 0, Qt::UserRole+1);
			}
			else {
				int vectorComponent = 0;
				Q_FOREACH(QString componentName, channel->componentNames()) {
					if(vectorComponent == mod->sourceVectorComponent() && mod->sourceDataChannel() == channel->name())
						initialIndex = channelList->count();
					channelList->addItem(QString("%1.%2").arg(channel->name(), componentName));
					channelList->setItemData(channelList->count()-1, channel->name(), Qt::UserRole+0);
					channelList->setItemData(channelList->count()-1, vectorComponent++, Qt::UserRole+1);
				}
			}
		}
	}

	if(mod->sourceDataChannel().isEmpty()) {
		// Select first available data channel if no one is selected yet.
		if(channelList->count() != 0) {
			mod->setSourceDataChannel(channelList->itemData(0, Qt::UserRole + 0).toString());
			mod->setSourceVectorComponent(channelList->itemData(0, Qt::UserRole + 1).toInt());
			// Auto adjust value range.
			mod->adjustRange();
			initialIndex = 0;
		}
	}
	else if(initialIndex < 0) {
		// Add a place-holder item if the selected channel does not exist anymore.
		initialIndex = channelList->count();
		channelList->addItem(tr("%1 (no longer available)").arg(mod->sourceDataChannel()));
		channelList->setItemData(initialIndex, mod->sourceDataChannel(), Qt::UserRole+0);
		channelList->setItemData(initialIndex, mod->sourceVectorComponent(), Qt::UserRole+1);
	}

	channelList->setCurrentIndex(initialIndex);
}

/******************************************************************************
* Updates the display for the color gradient.
******************************************************************************/
void ColorCodingModifierEditor::updateColorGradient()
{
	ColorCodingModifier* mod = static_object_cast<ColorCodingModifier>(editObject());
	if(!mod) return;

	// Create the color legend image.
	int legendHeight = 128;
	QImage image(1, legendHeight, QImage::Format_RGB32);
	for(int y=0; y<legendHeight; y++) {
		FloatType t = (FloatType)y / (FloatType)(legendHeight-1);
		Color color = mod->colorGradient()->valueToColor(1.0-t);
		image.setPixel(0, y, QColor(color).rgb());
	}
	colorLegendLabel->setPixmap(QPixmap::fromImage(image));

	// Select the right entry in the color gradient selector.
	PluginClassDescriptor* clazz = mod->colorGradient() ? mod->colorGradient()->pluginClassDescriptor() : NULL;
	colorGradientList->setCurrentIndex(colorGradientList->findData(qVariantFromValue((void*)clazz)));
}

/******************************************************************************
* This method is called when a reference target changes.
******************************************************************************/
bool ColorCodingModifierEditor::onRefTargetMessage(RefTarget* source, RefTargetMessage* msg)
{
	if(source == editObject() && msg->type() == REFTARGET_CHANGED) {
		// The modifier has changed -> update value shown in UI.
		ColorCodingModifier* mod = static_object_cast<ColorCodingModifier>(editObject());
		for(int index = 0; index < channelList->count(); index++) {
			if(channelList->itemData(index, Qt::UserRole + 0).toString() == mod->sourceDataChannel() &&
				channelList->itemData(index, Qt::UserRole + 1).toInt() == mod->sourceVectorComponent())
			{
				channelList->setCurrentIndex(index);
				break;
			}
		}
	}
	else if(source == editObject() && msg->type() == REFERENCE_FIELD_CHANGED && static_cast<ReferenceFieldMessage*>(msg)->field() == PROPERTY_FIELD_DESCRIPTOR(ColorCodingModifier, _colorGradient)) {
		updateColorGradient();
	}
	return PropertiesEditor::onRefTargetMessage(source, msg);
}

/******************************************************************************
* Is called when the user selects a Data Channel in the list box.
******************************************************************************/
void ColorCodingModifierEditor::onDataChannelSelected(int index)
{
	OVITO_ASSERT(!UNDO_MANAGER.isRecording());
	if(index < 0) return;
	ColorCodingModifier* mod = static_object_cast<ColorCodingModifier>(editObject());
	CHECK_OBJECT_POINTER(mod);

	UNDO_MANAGER.beginCompoundOperation(tr("Select data channel"));
	mod->setSourceDataChannel(channelList->itemData(index, Qt::UserRole + 0).toString());
	mod->setSourceVectorComponent(channelList->itemData(index, Qt::UserRole + 1).toInt());
	UNDO_MANAGER.endCompoundOperation();
}

/******************************************************************************
* Is called when the user selects a color gradient in the list box.
******************************************************************************/
void ColorCodingModifierEditor::onColorGradientSelected(int index)
{
	OVITO_ASSERT(!UNDO_MANAGER.isRecording());
	if(index < 0) return;
	ColorCodingModifier* mod = static_object_cast<ColorCodingModifier>(editObject());
	CHECK_OBJECT_POINTER(mod);

	PluginClassDescriptor* descriptor = (PluginClassDescriptor*)colorGradientList->itemData(index).value<void*>();
	if(descriptor == NULL) return;

	UNDO_MANAGER.beginCompoundOperation(tr("Change color gradient"));
	try {
		// Create an instance of the selected color gradient class.
		ColorCodingGradient::SmartPtr gradient = static_object_cast<ColorCodingGradient>(descriptor->createInstance());
		if(gradient) {
	        mod->setColorGradient(gradient.get());
		}
	}
	catch(const Exception& ex) {
		ex.showError();
		UNDO_MANAGER.currentCompoundOperation()->clear();
	}
	UNDO_MANAGER.endCompoundOperation();
}

/******************************************************************************
* Is called when the user presses the "Adjust Range" button.
******************************************************************************/
void ColorCodingModifierEditor::onAdjustRange()
{
	OVITO_ASSERT(!UNDO_MANAGER.isRecording());
	ColorCodingModifier* mod = static_object_cast<ColorCodingModifier>(editObject());
	CHECK_OBJECT_POINTER(mod);

	UNDO_MANAGER.beginCompoundOperation(tr("Adjust range"));
	mod->adjustRange();
	UNDO_MANAGER.endCompoundOperation();
}

/******************************************************************************
* Is called when the user presses the "Reverse Range" button.
******************************************************************************/
void ColorCodingModifierEditor::onReverseRange()
{
	ColorCodingModifier* mod = static_object_cast<ColorCodingModifier>(editObject());

	if(mod->startValueController() && mod->endValueController()) {
		UNDO_MANAGER.beginCompoundOperation(tr("Reverse range"));

		// Swap controllers for start and end value.
		FloatController::SmartPtr oldStartValue = mod->startValueController();
		mod->setStartValueController(mod->endValueController());
		mod->setEndValueController(oldStartValue.get());

		UNDO_MANAGER.endCompoundOperation();
	}
}

};	// End of namespace AtomViz
