/*
 *   Copyright 2009 - 2010 by Giulio Camuffo <giuliocamuffo@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as
 *   published by the Free Software Foundation; either version 2, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details
 *
 *   You should have received a copy of the GNU General Public
 *   License along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include "wolfmeter.h"

#include <QtGui/QGraphicsSceneMouseEvent>
#include "wolflabel.h"

WolfMeter::WolfMeter(QGraphicsWidget *parent)
    : Plasma::Meter(parent),
      m_label(new WolfLabel(this))
{
    m_label->setResizeFont(true, 1);
}

WolfMeter::~WolfMeter()
{
}

void WolfMeter::setPosition(int pos, int total)
{
    setMaximum(total);
    setValue(pos);
    if (total != 0) {
        qreal t = total;
        int n = (pos/t)*100;
        m_label->setText(QVariant(n).toString() + "%");
        updateLabel(size());
    }
}

void WolfMeter::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    qreal margin = (0.001338 * contentsRect().height() + 0.0002475) * contentsRect().width();
    if (event->buttons() == Qt::LeftButton) {
        int value = 0;
        if (meterType() == BarMeterHorizontal) {
            qreal x = event->pos().x();
            qreal width = contentsRect().width();
            qreal num = (x * 2 * margin - width * margin) / (width - 2 * margin);
            value = (((x + num) / width) * maximum());
        } else if (meterType() == BarMeterVertical) {
            qreal y = event->pos().y();
            qreal height = contentsRect().height();
            qreal num = (y * 2 * margin - height * margin) / (height - 2 * margin);
            value = maximum() - ((y + num) / height * maximum());
        }
        setValue(value);
        emit valueChanged(value);
    }
}

void WolfMeter::clear()
{
    setValue(0);
    m_label->clear();
}

void WolfMeter::resizeEvent(QGraphicsSceneResizeEvent *event)
{
    Plasma::Meter::resizeEvent(event);

    updateLabel(event->newSize());
}

void WolfMeter::updateLabel(QSizeF size)
{
    if (meterType() == BarMeterHorizontal) {
        if (size.height() > 7) {
            m_label->show();
            m_label->resize(size.width(), m_label->size().height());
            qreal y = (size.height() - m_label->size().height()) / 2.0;
            m_label->setPos(0, y);
        } else {
            m_label->hide();
        }
    } else if (meterType() == BarMeterVertical) {
        if (size.width() > 7) {
            m_label->show();
            m_label->resize(size.width(), m_label->size().height());
            qreal y = (size.height() - m_label->size().height()) / 2.0;
            m_label->setPos(0, y);
        } else {
            m_label->hide();
        }
    }
}

bool WolfMeter::contains(const QPointF& point) const
{
    QRectF rect = geometry();

    return ((point.x() >= rect.x()) and (point.x() <= (rect.x() + rect.width())) and
            (point.y() >= rect.y()) and (point.y() <= (rect.y() + rect.height())));
}

#include "wolfmeter.moc"
